<?php
// views/pages/api/facturas_recalcular_totales.php
// Recalcula y actualiza cantidad, totalfob, totalflete, totalseguro y total
// desde tb_detallefactura (detalle) + tb_datadai (valores económicos).

declare(strict_types=1);
session_start();

/* ===== Respuesta JSON & CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

/* ===== Preflight ===== */
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Método ===== */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido']);
  exit;
}

/* ===== Robustez: capturar fatales como JSON ===== */
error_reporting(E_ALL);
ini_set('display_errors', '0');
ob_start();
register_shutdown_function(function () {
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
    if (ob_get_level()) { ob_end_clean(); }
    http_response_code(500);
    echo json_encode([
      'success'=>false,
      'message'=>'Error fatal en servidor',
      'detail'=>($e['message'] ?? '') . ' @ ' . ($e['file'] ?? '') . ':' . ($e['line'] ?? 0)
    ]);
  }
});

/* ===== Conexión ===== */
require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db) { if (ob_get_level()) ob_end_clean(); echo json_encode(['success'=>false,'message'=>'Sin conexión a la base de datos']); exit; }
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'"); // Ecuador

/* ===== Entrada ===== */
$id_facturainfo = (int)($_POST['idfactura'] ?? 0);
if ($id_facturainfo <= 0) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'idfactura requerido.']);
  exit;
}

/*
  tb_datadai (campos relevantes):
    - fobunitario, fobitem (usa fobitem si viene, si no fobunitario)
    - transporte
    - seguro
    - valoraduana
  tb_detallefactura:
    - id_datadai, id_facturainfo, condicion
  Nota: La cantidad se basa en el número de detalles (COUNT(*)).
*/
$sql = "SELECT
          COUNT(*) AS cnt,
          COALESCE(SUM(COALESCE(dd.fobitem, dd.fobunitario)), 0) AS sum_fob,
          COALESCE(SUM(dd.transporte), 0) AS sum_flete,
          COALESCE(SUM(dd.seguro), 0) AS sum_seguro,
          COALESCE(SUM(dd.valoraduana), 0) AS sum_aduana
        FROM tb_detallefactura AS det
        INNER JOIN tb_datadai AS dd ON dd.iddatadai = det.id_datadai
        WHERE det.id_facturainfo = ? AND det.condicion = 1";

$stmt = mysqli_prepare($db, $sql);
if (!$stmt) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'No se pudo preparar la consulta de sumas.']);
  exit;
}
mysqli_stmt_bind_param($stmt, 'i', $id_facturainfo);
mysqli_stmt_execute($stmt);
$rs = mysqli_stmt_get_result($stmt);

$cantidad = 0; $sum_fob = 0.0; $sum_flete = 0.0; $sum_seguro = 0.0; $sum_aduana = 0.0;
if ($row = mysqli_fetch_assoc($rs)) {
  $cantidad   = (int)$row['cnt'];
  $sum_fob    = (float)$row['sum_fob'];
  $sum_flete  = (float)$row['sum_flete'];
  $sum_seguro = (float)$row['sum_seguro'];
  $sum_aduana = (float)$row['sum_aduana'];
}
mysqli_free_result($rs);
mysqli_stmt_close($stmt);

// total: si hay aduana (>0) úsala; si no, FOB+Flete+Seguro
$total = ($sum_aduana > 0) ? $sum_aduana : ($sum_fob + $sum_flete + $sum_seguro);

/* ===== Actualizar cabecera ===== */
$sqlU = "UPDATE tb_facturainfo
         SET cantidad = ?, totalfob = ?, totalflete = ?, totalseguro = ?, total = ?
         WHERE idfacturainfo = ?";
$stmtU = mysqli_prepare($db, $sqlU);
if (!$stmtU) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'No se pudo preparar el UPDATE de cabecera.']);
  exit;
}
mysqli_stmt_bind_param($stmtU, 'iddddi', $cantidad, $sum_fob, $sum_flete, $sum_seguro, $total, $id_facturainfo);
$ok = mysqli_stmt_execute($stmtU);
if (!$ok) {
  $err = mysqli_error($db);
  mysqli_stmt_close($stmtU);
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'No se pudo actualizar la cabecera: '.$err]);
  exit;
}
mysqli_stmt_close($stmtU);

/* ===== Éxito ===== */
if (ob_get_level()) ob_end_clean();
echo json_encode([
  'success'   => true,
  'recuento'  => [
    'cantidad'    => $cantidad,
    'totalfob'    => $sum_fob,
    'totalflete'  => $sum_flete,
    'totalseguro' => $sum_seguro,
    'total'       => $total,
    'valoraduana' => $sum_aduana
  ]
]);
