<?php
declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== Cabeceras JSON + CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Methods: GET, OPTIONS');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Manejo de errores como JSON ===== */
ini_set('display_errors', '0');
error_reporting(E_ALL);
set_error_handler(function($s,$m,$f,$l){
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'PHP error','detail'=>"$m in $f:$l"], JSON_UNESCAPED_UNICODE);
  exit;
});
set_exception_handler(function($e){
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Exception','detail'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
  exit;
});

/* ===== Conexión ===== */
require_once __DIR__ . '/../conex.php';
if (!isset($conn) || !$conn instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión a la base de datos']); exit;
}
$conn->set_charset('utf8mb4');

/* ===== Parámetros =====
   - id_ubicacion (int)   -> filtra un área específica (opcional)
   - solo (string)        -> 'libres' | 'ocupados' (opcional)
   - order (string)       -> 'fila_col' (default) | 'ubicacion_fila_col'
*/
$id_ubicacion = isset($_GET['id_ubicacion']) ? (int)$_GET['id_ubicacion'] : 0;
$solo         = isset($_GET['solo']) ? strtolower(trim((string)$_GET['solo'])) : '';
$order        = isset($_GET['order']) ? strtolower(trim((string)$_GET['order'])) : 'fila_col';

/* ===== Base SQL: todos los andenes + ocupación vigente (si existe) ===== */
$sql = "
SELECT
  a.id                         AS id_anden,
  a.fila,
  a.columna,
  a.etiqueta,
  a.activo                     AS flag_activo_anden,
  a.id_ubicacion,
  u.detalle_ubicacion,

  -- Ocupación vigente si existe
  da.id_datadai,
  da.fecha_i,

  d.iddatadai                  AS id_vehiculo,
  d.chasis,
  d.marca,
  d.modelo,
  d.color
FROM tb_anden AS a
LEFT JOIN tb_ubicacion AS u
  ON u.idubicacion = a.id_ubicacion
LEFT JOIN (
  /* última ocupación ACTIVA por andén (condicion=1) */
  SELECT x.*
  FROM tb_datadai_anden x
  INNER JOIN (
    SELECT id_anden, MAX(fecha_i) AS max_fecha
    FROM tb_datadai_anden
    WHERE condicion = 1
    GROUP BY id_anden
  ) m
  ON m.id_anden = x.id_anden AND m.max_fecha = x.fecha_i
  WHERE x.condicion = 1
) AS da
  ON da.id_anden = a.id
LEFT JOIN tb_datadai AS d
  ON d.iddatadai = da.id_datadai
WHERE a.condicion = 1
";

/* ===== Filtros dinámicos ===== */
$types = '';
$params = [];

if ($id_ubicacion > 0) {
  $sql .= " AND a.id_ubicacion = ? ";
  $types .= 'i';
  $params[] = $id_ubicacion;
}
if ($solo === 'libres') {
  $sql .= " AND da.id_datadai IS NULL ";
} elseif ($solo === 'ocupados') {
  $sql .= " AND da.id_datadai IS NOT NULL ";
}

/* ===== Orden ===== */
switch ($order) {
  case 'ubicacion_fila_col':
    $sql .= " ORDER BY u.detalle_ubicacion, a.fila, a.columna ";
    break;
  case 'fila_col':
  default:
    $sql .= " ORDER BY a.fila, a.columna ";
    break;
}

/* ===== Preparar y ejecutar ===== */
$stmt = $conn->prepare($sql);
if (!$stmt) throw new RuntimeException('Prepare failed: '.$conn->error);
if ($types !== '') $stmt->bind_param($types, ...$params);
$stmt->execute();
$res = $stmt->get_result();

/* ===== Armar respuesta para el GRID =====
   Estructura por item:
   {
     id_anden, fila, columna, etiqueta, id_ubicacion, ubicacion,
     ocupado (bool), estado ('OCUPADO'|'LIBRE'), dias_ocupado (int|null), fecha_i (Y-m-d H:i:s|null),
     vehiculo: { id, chasis, marca, modelo, color } | null
   }
*/
$data = [];
$now = new DateTimeImmutable('now');

while ($row = $res->fetch_assoc()) {
  $tieneOcupacion = !empty($row['id_datadai']); // si hay id_datadai vigente
  $fechaI = $row['fecha_i'] ?? null;

  $diasOcupado = null;
  if ($tieneOcupacion && $fechaI) {
    try {
      $fi = new DateTimeImmutable($fechaI);
      $diasOcupado = (int)$fi->diff($now)->format('%a');
    } catch (Throwable $e) { $diasOcupado = null; }
  }

  $data[] = [
    'id_anden'       => (int)$row['id_anden'],
    'fila'           => (int)$row['fila'],
    'columna'        => (int)$row['columna'],
    'etiqueta'       => $row['etiqueta'],
    'id_ubicacion'   => (int)$row['id_ubicacion'],
    'ubicacion'      => $row['detalle_ubicacion'],

    'ocupado'        => $tieneOcupacion ? 1 : 0,
    'estado'         => $tieneOcupacion ? 'OCUPADO' : 'LIBRE',
    'dias_ocupado'   => $diasOcupado,
    'fecha_i'        => $tieneOcupacion ? $fechaI : null,

    'vehiculo'       => $tieneOcupacion ? [
      'id'     => isset($row['id_vehiculo']) ? (int)$row['id_vehiculo'] : null,
      'chasis' => $row['chasis'] ?? null,
      'marca'  => $row['marca'] ?? null,
      'modelo' => $row['modelo'] ?? null,
      'color'  => $row['color'] ?? null,
    ] : null
  ];
}

echo json_encode(['success'=>true,'count'=>count($data),'data'=>$data], JSON_UNESCAPED_UNICODE);
